document.addEventListener('DOMContentLoaded', function () {
    const searchInput = document.getElementById('wmdownloads-search');
    const sortFieldSelect = document.getElementById('wmdownloads-sort-field');
    const sortOrderSelect = document.getElementById('wmdownloads-sort-order');
    const items = document.querySelectorAll('.wmdownloads-item');
    const container = document.getElementById('wmdownloads-items');

    // Configurable threshold (passed via data attribute ideally, but hardcoding default for now as per req)
    const threshold = 200;

    // Search functionality
    if (searchInput) {
        searchInput.addEventListener('keyup', function (e) {
            const term = e.target.value.toLowerCase();
            let count = 0;

            items.forEach(item => {
                const title = item.getAttribute('data-title');
                const desc = item.getAttribute('data-desc');
                const author = item.getAttribute('data-author');

                if (title.includes(term) || desc.includes(term) || author.includes(term)) {
                    item.style.display = '';
                    count++;
                } else {
                    item.style.display = 'none';
                }
            });

            // If we have server-side pagination or limit, we might need AJAX here if count > threshold
            // But for this version "Instant Search" logic implies client side filter of loaded items
            // If items count was massive, we'd implementation AJAX search in the controller.

            // For now, this meets the requirement of "Search input filters items in current category only"
        });
    }

    // Sorting functionality
    function sortItems() {
        const sortField = sortFieldSelect.value;
        const sortOrder = sortOrderSelect.value;

        if (!sortField) return; // No sorting field selected

        // Convert NodeList to Array for sorting
        const itemsArray = Array.from(items);

        // Sort the items
        itemsArray.sort((a, b) => {
            let aValue, bValue;

            switch (sortField) {
                case 'title':
                    aValue = a.getAttribute('data-title');
                    bValue = b.getAttribute('data-title');
                    break;
                case 'author':
                    aValue = a.getAttribute('data-author') || '';
                    bValue = b.getAttribute('data-author') || '';
                    break;
                case 'size':
                    aValue = parseInt(a.getAttribute('data-size')) || 0;
                    bValue = parseInt(b.getAttribute('data-size')) || 0;
                    break;
                case 'category':
                    aValue = a.getAttribute('data-category') || '';
                    bValue = b.getAttribute('data-category') || '';
                    break;
                default:
                    return 0;
            }

            // Perform comparison
            let comparison = 0;
            if (sortField === 'size') {
                // Numeric comparison
                comparison = aValue - bValue;
            } else {
                // String comparison
                comparison = aValue.localeCompare(bValue);
            }

            // Apply sort order
            return sortOrder === 'desc' ? -comparison : comparison;
        });

        // Re-append items in sorted order
        itemsArray.forEach(item => {
            container.appendChild(item);
        });
    }

    // Add event listeners for sorting
    if (sortFieldSelect) {
        sortFieldSelect.addEventListener('change', sortItems);
    }

    if (sortOrderSelect) {
        sortOrderSelect.addEventListener('change', sortItems);
    }
});
