<?php
namespace Joomla\Component\Wmdownloads\Administrator\Controller;

use Joomla\CMS\MVC\Controller\BaseController;
use Joomla\CMS\Factory;
use Joomla\CMS\Filesystem\File;
use Joomla\CMS\Language\Text;

defined('_JEXEC') or die;

class ImportController extends BaseController
{
    public function import()
    {
        // Check for request forgeries
        $this->checkToken();

        $app = Factory::getApplication();
        $input = $app->getInput();
        $file = $input->files->get('import_file');

        if (empty($file) || $file['error'] !== 0) {
            $app->enqueueMessage(Text::_('No file selected or upload error'), 'error');
            $app->redirect('index.php?option=com_wmdownloads&view=import');
            return;
        }

        // Check extension
        $ext = strtolower(File::getExt($file['name']));
        if ($ext !== 'csv') {
            $app->enqueueMessage(Text::_('Invalid file type. Please upload a CSV file.'), 'error');
            $app->redirect('index.php?option=com_wmdownloads&view=import');
            return;
        }

        // Process CSV
        $content = file_get_contents($file['tmp_name']);

        // Detect and convert encoding if needed (assume CP1252/Windows-1252 if not UTF-8)
        // This fixes "Incorrect string value" database errors
        if (!mb_check_encoding($content, 'UTF-8')) {
            $content = mb_convert_encoding($content, 'UTF-8', 'Windows-1252');
        }

        // Save back to temp file
        file_put_contents($file['tmp_name'], $content);

        $handle = fopen($file['tmp_name'], 'r');
        if ($handle === false) {
            $app->enqueueMessage(Text::_('Could not open file'), 'error');
            $app->redirect('index.php?option=com_wmdownloads&view=import');
            return;
        }

        $db = Factory::getContainer()->get('DatabaseDriver');
        $imported = 0;
        $errors = [];
        $row = 0;

        $storagePath = JPATH_ROOT . '/media/com_wmdownloads/files/';

        while (($data = fgetcsv($handle, 1000, ",")) !== FALSE) {
            // Check for empty rows
            if (empty($data) || (count($data) === 1 && empty($data[0]))) {
                continue;
            }

            $row++;
            // Skip header
            if ($row === 1) {
                continue;
            }

            // Map columns: 0:filename, 1:title, 2:description, 3:category_alias, 4:author, 5:created_date, 6:hits
            // Clean filename of any invisible characters/spaces
            $filenameRaw = trim($data[0] ?? '');
            $filename = str_replace(array("\r", "\n", "\t", "\0", "\x0B"), '', $filenameRaw);

            if (empty($filename)) {
                $errors[] = "Row $row: Filename is missing.";
                continue;
            }

            // Check if file exists in storage
            if (!file_exists($storagePath . $filename)) {
                $errors[] = "Row $row: File '$filename' not found in storage directory.";
                continue;
            }

            // Prepare Data
            $title = !empty($data[1]) ? $data[1] : pathinfo($filename, PATHINFO_FILENAME);
            $desc = $data[2] ?? '';
            $catAlias = !empty($data[3]) ? $data[3] : '';
            $author = $data[4] ?? '';
            $created = !empty($data[5]) ? Factory::getDate($data[5])->toSql() : Factory::getDate()->toSql();
            $hits = !empty($data[6]) ? (int) $data[6] : 0;

            // Resolve Category ID from Alias
            $catid = 0;
            if (!empty($catAlias)) {
                $extension = 'com_content';
                $query = $db->getQuery(true)
                    ->select($db->quoteName('id'))
                    ->from($db->quoteName('#__categories'))
                    ->where($db->quoteName('alias') . ' = :alias')
                    ->where($db->quoteName('extension') . ' = :extension')
                    ->bind(':alias', $catAlias)
                    ->bind(':extension', $extension); // Using com_content categories
                $db->setQuery($query);
                try {
                    $catid = (int) $db->loadResult();
                    if (!$catid) {
                        // Category alias not found, maybe warn or skip?
                        // For now, logging error.
                        $errors[] = "Row $row: Category alias '$catAlias' not found.";
                        continue;
                    }
                } catch (\Exception $e) {
                    $errors[] = "Row $row: Database error lookup category.";
                    continue;
                }
            }

            // Get file stats
            $fileStats = stat($storagePath . $filename);
            $size = $fileStats['size'];
            $fileExt = strtolower(pathinfo($filename, PATHINFO_EXTENSION));

            // Simple mime mapping or use finfo (preferred if available)
            $finfo = finfo_open(FILEINFO_MIME_TYPE);
            $mime = finfo_file($finfo, $storagePath . $filename);
            finfo_close($finfo);

            // Type Code Map
            $typeMap = [
                'pdf' => 'pdf',
                'doc' => 'word',
                'docx' => 'word',
                'xls' => 'excel',
                'xlsx' => 'excel',
                'ppt' => 'powerpoint',
                'pptx' => 'powerpoint',
                'txt' => 'text',
                'rtf' => 'text',
                'csv' => 'csv'
            ];
            $typeCode = $typeMap[$fileExt] ?? 'text';

            // Insert into DB
            $item = new \stdClass();
            $item->item_mode = 'file';
            $item->catid = $catid;
            $item->title = $title;
            // Generate alias
            $item->alias = \Joomla\CMS\Filesystem\File::makeSafe($title) . '-' . time(); // Unique alias
            $item->description = $desc;
            $item->author = $author;
            $item->created = $created;
            $item->hits = $hits;
            $item->file_path = $filename; // We store just filename if it's in the root of storage
            $item->original_filename = $filename;
            $item->ext = $fileExt;
            $item->mime = $mime;
            $item->size_bytes = $size;
            $item->type_code = $typeCode;
            $item->state = 1; // Published by default
            $item->access = 1; // Public
            $item->language = '*';
            $item->created_by = Factory::getUser()->id;

            try {
                $db->insertObject('#__wmdownloads_items', $item);
                $imported++;
            } catch (\Exception $e) {
                $errors[] = "Row $row: Database error - " . $e->getMessage();
            }
        }
        fclose($handle);

        $msgType = 'message';
        $msg = "Imported $imported items successfully.";
        if (count($errors) > 0) {
            $msgType = 'warning';
            $msg .= " With errors: <br>" . implode('<br>', $errors);
        }

        $app->enqueueMessage($msg, $msgType);
        $app->redirect('index.php?option=com_wmdownloads&view=items');
    }

    public function downloadTemplate()
    {
        $filename = 'import_template.csv';
        $data = "filename,title,description,category_alias,author,created_date,hits\n";
        $data .= "example_file.pdf,Example Title,Description here,news,Author Name,2026-01-01 12:00:00,0";

        // Clean buffer
        if (ob_get_contents()) {
            ob_end_clean();
        }

        header('Content-Type: text/csv');
        header('Content-Disposition: attachment; filename="' . $filename . '"');
        header('Content-Length: ' . strlen($data));
        echo $data;
        exit;
    }

    public function cancel()
    {
        $this->setRedirect('index.php?option=com_wmdownloads&view=items');
    }
}
